""" 
    File Name: main.py
    Author: Franziska Lange and Joris Löttgen
    Information:
    Application to view and save sensor data through periodical syncing with measurement unit.
    If executed as python file, modules pyserial, matplotlib and pandas need to be installed via pip.
"""

import time
import tkinter as tk
from tkinter import ttk
from tkinter import messagebox
from tkinter import font as tkFont

from PIL import Image, ImageTk

from matplotlib.backends.backend_tkagg import (FigureCanvasTkAgg, NavigationToolbar2Tk)
from matplotlib.figure import Figure
import matplotlib.dates as mdates

import datetime as dt

import serial
import serial.tools.list_ports
import pandas as pd

import webbrowser

import tkinter.filedialog as fd


# Open a serial connection to the measurement unit

def connect():
    global serialTerminal

    connect_dialog()

    if serialPort != "":
        serialTerminal.port = serialPort
        serialTerminal.open()

        if serialTerminal.isOpen():
            messagebox.showinfo("Serial Connection", "Successfully connected!")
            menu_unit.entryconfig("Start New", state="normal")
            menu_unit.entryconfig("Sync Data", state="normal")
            menu_unit.entryconfig("Set Interval", state="normal")
            menu_unit.entryconfig("Connect", state="disabled")
            menu_unit.entryconfig("Eject", state="normal")
        else:
            messagebox.showerror("Serial Connection", "Connection failed!")


# Close the serial connection

def eject():
    global serialTerminal
    global serialPort

    with open("resources\\variables.txt", "w") as f:
        f.writelines(str(dt.datetime.now()))
    send("x")
    while serialTerminal.out_waiting > 0:
        pass

    serialTerminal.close()
    serialPort = ""
    messagebox.showinfo("Serial Connection", "Serial Connection terminated\n You may remove the unit now")
    menu_unit.entryconfig("Start New", state="disabled")
    menu_unit.entryconfig("Sync Data", state="disabled")
    menu_unit.entryconfig("Set Interval", state="disabled")
    menu_unit.entryconfig("Connect", state="normal")
    menu_unit.entryconfig("Eject", state="disabled")


# Asking user to select the com-port connected to the unit to open a serial connection

def connect_dialog():
    global serialPort

    ask_com = tk.Toplevel()
    ask_com.grab_set()
    ask_com.configure(bg=my_blue)
    ask_com.geometry("600x250")
    ask_com.iconbitmap("resources\\igem.ico")
    ask_com.wm_title("Select COM")

    frame_com = ttk.Frame(ask_com)

    selected = tk.StringVar(ask_com)

    descriptions = []
    devices = []

    for port in list(serial.tools.list_ports.comports()):
        descriptions.append(port.description)
        devices.append(port.device)

    if not devices:
        ask_com.destroy()
        messagebox.showerror(title="Connection Error", message="No device found!")
        return

    port_input = ttk.OptionMenu(frame_com, selected, descriptions[0],  *descriptions, style="dialog.TMenubutton")
    port_input.grid(row=0, column=0, pady=30)

    ok = ttk.Button(frame_com, text="Okay", command=lambda: return_connect_dialog(ask_com, selected, descriptions, devices), style="dialog.TButton")
    ok.grid(row=1, column=0, pady=30)

    frame_com.place(relx=0.5,rely=0.5,anchor="c")

    root.wait_window(ask_com)


def return_connect_dialog(window, name, names, ports):
    global serialPort

    index = names.index(name.get())
    serialPort = ports[index]
    window.destroy()


# Send single byte command: s=sync, d=delete, i=interval, c=close, x=sync finished

def send(data):
    if type(data) is int:
        serialTerminal.write(data)
    else:
        serialTerminal.write(data.encode('utf-8'))


# Start new measurement tracking

def start_new():
    global data
    global x_values

    send('d')

    empty_data_df = pd.DataFrame([], columns=["tem", "hum", "o2", "co2", "date"])
    empty_data_df.to_csv("resources\\data.csv", mode="w", header=True, index=False)
    data = empty_data_df
    x_values = []
    update_plots()

    
# Sync with measurement unit, get captured datapoints, delete them on the unit, add estimated time data, update plots

def sync_data():
    global serialTerminal
    global data
    global last_connection
    global x_values

    send('s')

    while serialTerminal.in_waiting == 0:
        pass

    previous = 0
    current = serialTerminal.in_waiting

    while previous != current:
        previous = current
        time.sleep(0.01)
        current = serialTerminal.in_waiting

    message = ""
    while serialTerminal.in_waiting > 0:
        message += str(serialTerminal.read(), 'utf-8')

    new_data_enc = message.split(';')
    new_data_dec = []

    start_time = last_connection
    interval = int(new_data_enc.pop())

    for index in range(len(new_data_enc) - 1):
        new_data_dec.append(new_data_enc[index].split(":"))
        new_data_dec[index].append(str(start_time))
        x_values.append(start_time)
        start_time = start_time + dt.timedelta(seconds=interval)

    new_data_df = pd.DataFrame(new_data_dec, columns=["tem", "hum", "o2", "co2", "date"])

    new_data_df.to_csv("resources\\data.csv", mode="a", header=False, index=False)

    data = pd.read_csv("resources\\data.csv", sep=',', header=0)
    update_plots()

    send('d')


# Get new interval from user and send to unit

def interval_dialog():

    ask_interval = tk.Toplevel()
    ask_interval.grab_set()
    ask_interval.configure(bg=my_blue)
    ask_interval.geometry("600x140")
    ask_interval.iconbitmap("resources\\igem.ico")
    ask_interval.wm_title("Select Interval")

    frame_interval = ttk.Frame(ask_interval)

    interval = tk.StringVar(frame_interval)
    interval_entry = ttk.Entry(frame_interval, takefocus=True, textvariable=interval, style="dialog.TEntry")
    interval_entry.grid(row=0, column=0, columnspan=2, padx=5)

    selected = tk.StringVar(frame_interval)
    units = [
        "Second(s)",
        "Minute(s)",
        "Hour(s)",
        "Day(s)"
    ]

    unit_input = ttk.OptionMenu(frame_interval, selected, units[1], *units, style="dialog.TMenubutton")
    unit_input.grid(row=0, column=2, padx=10)

    ok = ttk.Button(frame_interval, text="Okay", command=lambda: return_interval_dialog(ask_interval, interval, selected), style="dialog.TButton")
    ok.grid(row=0, column=3, padx=35)

    frame_interval.place(relx=0.5,rely=0.5,anchor="c")

    root.wait_window(ask_interval)


def return_interval_dialog(window, interval, unit):
    enc_interval = int(interval.get())
    dec_interval = 0
    if unit == "Minute(s)":
        dec_interval = enc_interval * 60
    elif unit == "Hour(s)":
        dec_interval = enc_interval * 3600
    elif unit == "Day(s)":
        dec_interval = enc_interval * 86400
    else:
        dec_interval = enc_interval

    send('i')
    for c in str(dec_interval):
        send(c)
    send('c')

    messagebox.showinfo("Serial Connection", "Set new interval of " + str(enc_interval) + " " + unit)
    window.destroy()


# Update data in all plots, redraw line

def update_plots():
    global tab_data_list

    for tab_data in tab_data_list:
        (tab_data[4]).remove()
        tab_data[4], = tab_data[5].plot(x_values, data[tab_data[1]])
        tab_data[4].set_color(my_blue)
        tab_data[3].draw()


# Handling closing of an advanced view

# def on_closing_advance(i):
#     global advanced
#     advanced[i].destroy()
#     advanced[i] = None


# Open instance of an advanced view

# def open_advanced():
#     global advanced

#     index = 0

#     if not any(elem is None for elem in advanced):
#         index = len(advanced)
#         advanced.append(tk.Toplevel())
#     else:
#         for i in range(len(advanced)):
#             if advanced[i] is None:
#                 advanced[i] = tk.Toplevel()
#                 index = i
#                 break
    
#     advanced[index].configure(bg=my_blue)
#     advanced[index].geometry("800x500")
#     advanced[index].iconbitmap("resources\\igem.ico")
#     advanced[index].wm_title("Advanced View #" + str(index + 1))
#     advanced[index].protocol("WM_DELETE_WINDOW", lambda: on_closing_advance(index))

#     #adv_frame = ttk.Frame(advanced[index])


# Closing all Advanced Popups currently open

# def close_all_advanced():
#     global advanced

#     for adv in advanced:
#         adv.destroy()
#     advanced = []


# Import csv data to replace or append to existing data

def import_data():
    global data

    filetype = [("csv file(*.csv)","*.csv")]
    file = fd.askopenfilename(filetypes = filetype, defaultextension = filetype)

    if file != "":
        imported_data = pd.read_csv(file, sep=',', header=0)
        imported_data.to_csv("resources\\data.csv", mode="w", header=True, index=False)
        data = pd.read_csv("resources\\data.csv", sep=',', header=0)
        update_plots()

        messagebox.showinfo("Import Data", "Imported data into storage" )


# Export csv data

def export_data():
    global data

    filetype = [("csv file(*.csv)","*.csv")]
    exp = pd.read_csv("resources\\data.csv", sep=',', header=0)
    file = fd.asksaveasfilename(filetypes = filetype, defaultextension = filetype)

    if file != "":
        with open(file,"w") as f:
            f.write(exp.to_csv(header=True, index=False))

# Before exiting, close serial connection, restart measurements, save permanent variables

def on_closing():
    global last_connection
    if messagebox.askokcancel("Quit", "Do you want to quit?"):

        # close_all_advanced()
        if serialTerminal.isOpen():
            eject()
        root.destroy()





# Create main window

root = tk.Tk()
root.geometry("1000x600")
root.minsize(975, 550)
root.title('Sensor Data Visualization')
root.iconbitmap("resources\\igem.ico")
root.protocol("WM_DELETE_WINDOW", on_closing)


# Array for references to Adanced instances

advanced = []


# Create variables used in serial communication

serialTerminal = serial.Serial(port=None, baudrate=9600, bytesize=8, timeout=10, stopbits=serial.STOPBITS_ONE)
serialPort = ""


# Read permanent variables from txt-file

with open("resources\\variables.txt", "r") as f:
    last_connection = dt.datetime.strptime(f.readline(), "%Y-%m-%d %H:%M:%S.%f")


# Set standard styles

my_blue = "#0A0A2B"
my_yellow = "#FFB100"

trebuchet = tkFont.Font(family="Trebuchet MS", size=20, weight="bold")
trebuchet_small = tkFont.Font(family="Trebuchet MS", size=14)
trebuchet_small_bold = tkFont.Font(family="Trebuchet MS", size=14, weight="bold")
trebuchet_smaller_bold = tkFont.Font(family="Trebuchet MS", size=12, weight="bold")
trebuchet_big = tkFont.Font(family="Trebuchet MS", size=23, weight="bold")

style = ttk.Style()

style.theme_create( "standard", parent="default", settings={
        "TNotebook": {
            "configure": {"background": my_blue,
                          "tabposition": 'wn',
                          "tabmargins": [5, 5, -4, 5],
                          "padding": [10, 30, 30, 30],
                          "borderwidth": 0
                          } 
        },
        "TNotebook.Tab": {
            "configure": {"padding": [5, 20], 
                          "background": my_blue,
                          "foreground": my_yellow,
                          "font": trebuchet,
                          "borderwidth": 0,
                          "width" : 12
                          },
            "map":       {"background": [("selected", my_yellow)],
                          "foreground": [("selected", my_blue)],
                          "font": [("selected", trebuchet_big)]
                          } 
        },
        "TFrame": {
            "configure": {
                          "background": my_blue
                          }
        }
    })

style.theme_use("standard")

style.configure("dialog.TButton", foreground=my_blue, background=my_yellow, font=trebuchet_small_bold, padding=[40,5])
style.configure("dialog.TMenubutton", foreground=my_blue, background=my_yellow, font=trebuchet_smaller_bold, padding=[10,0])
style.configure("dialog.TEntry", padding=[6,6], borderwidth=0)

# Retrieve data from csv-file

data = pd.read_csv("resources\\data.csv", sep=',', header=0)


# Start page frame

frame = ttk.Frame(root)
frame.pack(fill=tk.BOTH, expand=True)


# Building tabs

tabs = ttk.Notebook(frame)


# Info Tab

im = Image.open("resources\\rarecycle.png")
ph = ImageTk.PhotoImage(im)

info_frame = ttk.Frame(tabs)
tabs.add(info_frame, image=ph)

logo_im = Image.open("resources\\igem.ico")
resized_logo = logo_im.resize([150, 150])
logo_ph = ImageTk.PhotoImage(resized_logo)

logo = tk.Label(info_frame, image=logo_ph, background=my_blue)
logo.pack(pady=40)

info_text = tk.Text(info_frame, height=6, background=my_blue, foreground=my_yellow, font=trebuchet_small, borderwidth=0)
info_text.tag_configure("center", justify='center')

info_text.insert('1.0', "This software has been developed within the international iGEM competition\n\n" + 
                 "as part of the hardware project of the iGEM Team Aachen 2023.\n\n" + 
                 "For more information about our project go to our project website.\n\n")

info_text.tag_add("center", "1.0", "end")
info_text.configure(state="disabled")
info_text.pack()

link = ttk.Button(info_frame, text="To project page", command=lambda: webbrowser.open_new_tab("https://2023.igem.wiki/aachen/"), style="dialog.TButton")
link.pack(pady=20)


# Plot Tabs

x_label = "Time [Days]"
x_values = []
for date in data['date']:
    x_values.append(dt.datetime.strptime(date, "%Y-%m-%d %H:%M:%S.%f"))

tab_data_list = [["Temperature", "tem", "Temperature [°]", None, None, None],
            ["Humidity", "hum", "Humidity [%]", None, None, None],
            ["O2", "o2", "O2 Concentration [%]", None, None, None],
            ["CO2", "co2", "CO2 Concentration [PPM]", None, None, None]]

for tab_data in tab_data_list:
    fr = ttk.Frame(tabs)
    tabs.add(fr, text=tab_data[0])

    fig = Figure(figsize=(5, 4), dpi=100)
    tab_data[5] = fig.add_subplot()
    fig.subplots_adjust(left=0.16, bottom=0.24)
    tab_data[4], = tab_data[5].plot(x_values, data[tab_data[1]])
    tab_data[4].set_color(my_blue)
    tab_data[5].set_xlabel(x_label)
    tab_data[5].set_ylabel(tab_data[2])
    if not x_values:
        now = dt.datetime.now()
        tab_data[5].set_xlim([now - dt.timedelta(days=7), now])
    else:
        tab_data[5].set_xlim([x_values[-1] - dt.timedelta(days=7), x_values[-1]])
    tab_data[5].xaxis.set_major_locator(mdates.DayLocator(bymonthday=range(1,32)))
    tab_data[5].xaxis.set_minor_locator(mdates.HourLocator(interval=4))
    tab_data[5].xaxis.set_major_formatter(mdates.DateFormatter("%d.%m.%Y"))
    # Rotates and right-aligns the x labels so they don't crowd each other.
    for label in tab_data[5].get_xticklabels(which='major'):
        label.set(rotation=30, horizontalalignment='right')

    tab_data[3] = FigureCanvasTkAgg(fig, master=fr)
    tab_data[3].draw()
    
    toolbar = NavigationToolbar2Tk(tab_data[3], fr, pack_toolbar=False)
    toolbar.config(height=50, padx=5)
    toolbar.update()

    toolbar.pack(fill=tk.X, expand=False)
    tab_data[3].get_tk_widget().pack(fill=tk.BOTH, expand=True)


# tabs.bind("<<NotebookTabChanged>>", on_tab_change)        not needed, could be used in the future
tabs.pack(fill=tk.BOTH, expand=True)
tabs.select(1)

# Create Menubar

menubar = tk.Menu(root)

menu_unit = tk.Menu(menubar, tearoff=0)
menubar.add_cascade(label="Serial Connection", menu=menu_unit)
menu_unit.add_command(label="Start New", command=start_new, state="disabled")
menu_unit.add_command(label="Sync Data", command=sync_data, state="disabled")
menu_unit.add_command(label="Set Interval", command=interval_dialog, state="disabled")
menu_unit.add_separator()
menu_unit.add_command(label="Connect", command=connect)
menu_unit.add_command(label="Eject", command=eject, state="disabled")

# menu_view = tk.Menu(menubar, tearoff=0)
# menubar.add_cascade(label="View", menu=menu_view)
# menu_view.add_command(label="New Advanced", command=open_advanced)
# menu_view.add_command(label="Close All Advanced", command=close_all_advanced)

menu_data = tk.Menu(menubar, tearoff=0)
menubar.add_cascade(label="Data", menu=menu_data)
menu_data.add_command(label="Import", command=import_data)
menu_data.add_command(label="Export", command=export_data)


root.config(menu=menubar)

# Start application

root.mainloop()

