import streamlit as st
from mycoflux import adsorption, desorption, growth, preprocessing
import plotly.express as px
import pandas as pd

st.set_page_config(layout="wide")

@st.cache_data
def adsorp_plot(V_all,X_REE_in,N_in,n_S,V_S,k_ad,B_inf,M_S,rho_S):
    tsim, vartable = adsorption(V_all,X_REE_in,N_in,n_S,V_S,k_ad,B_inf,M_S,rho_S)
    data = pd.DataFrame(vartable)
    data.insert(0, 'Time', tsim)
    fig = px.line(data, x="Time", y=["n_B","n_BS","n_prod","V_prod"])
    return fig, data

@st.cache_data
def desorp_plot(V_all,n_REE_bound,N_in,n_D,V_D,k_de,B_inf,M_D,rho_D):
    tsim, vartable = desorption(V_all,n_REE_bound,N_in,n_D,V_D,k_de,B_inf,M_D,rho_D)
    data = pd.DataFrame(vartable)
    data.insert(0, 'Time', tsim)
    fig = px.line(data, x="Time", y=["n_B","n_BS","n_prod","V_prod"])
    return fig, data

@st.cache_data
def growth_plot(duration,X_bio_0,V_Tray,eps_Tray,mu_max,K,rho_Sub,Y_XS):
    tsim, vartable = growth(duration,X_bio_0,V_Tray,eps_Tray,mu_max,K,rho_Sub,Y_XS)
    data = pd.DataFrame(vartable)
    data.insert(0, 'Time', tsim)
    fig = px.line(data, x="Time", y=["X_bio","S","mu"])
    return fig, data


biomasstracker = 0.01
boundREEtracker = 0.0

st.title("MycoFlux Geometry")
col1, col2 = st.columns(2)

# Inputs to preprocession
with col1:
    V_Tray = st.number_input("V_Tray: Volume of Extraction Space in MycoFlux [L]", min_value=25, max_value=100, step=25) /1000

with col2:
    eps_Tray = st.number_input("eps_Tray: Average Porosity of Extraction Space [%]", min_value=20, max_value=50, step=1) /100


st.title("Mycelial Growth")
col1, col2 = st.columns(2)

with col1:
    # Set up
    X_bio_0 = st.number_input("X_bio_0: Inital Biomass [kg]", min_value=0.01, max_value=999.9, step=0.01)
    duration = st.number_input("t_growth: Growth Duration [h]", min_value=0.1, max_value=200.0, step=0.5) * 3600

# Kinetics
    mu_max = st.number_input("mu_max: Maximum Biomass Growth Rate [1/h]", min_value=0.15, max_value=0.4, step=0.01) / 3600

with col2:
    K = st.number_input("K: Monod Constant [g/L]", min_value=0.3, max_value=0.5, step=0.01) / 1000 * 1000
    rho_Sub = st.number_input("rho_Sub: Density of Substrate [kg/m^3]", min_value=95.0, max_value=150.0, step=0.5)
    Y_XS = st.number_input("Y_XS: Biomass from Substrate Yield [g/g]", min_value=0.3, max_value=0.6, step=0.01)

growth_fig, gr_data = growth_plot(duration,X_bio_0,V_Tray,eps_Tray,mu_max,K,rho_Sub,Y_XS)

with col1:
    st.subheader("Variable Plots")
    st.plotly_chart(growth_fig)

df_grow = gr_data.iloc[-1]
df_grow = df_grow.to_frame().reset_index()
df_grow.columns = ["Variable", "Value"]
df_grow = df_grow.drop(index=0)
df_grow["Unit"] = ['kg','kg','1/h']
with col2:
    st.subheader("Final Values")
    st.table(df_grow)

biomasstracker = gr_data["X_bio"][len(gr_data["X_bio"])-1]

st.title("Adsorption")
col1, col2, col3 = st.columns(3)

# Inputs directly to model

with col1:
    V_ad = st.number_input("V_ad: Amount of Treated Solution [L]", min_value=1.0, max_value=25.0, step=0.5) / 1000
    X_REE_in = st.number_input("X_REE_in: Incoming Load of REE on Solution [mol/mol]", min_value=0.0000, max_value=10.0000, step=0.01)

with col2:
    V_in = st.number_input("V_in: Volumetric Flow Rate of Solution into Tray [L/h]", min_value=1200, max_value=1500, step=10) /1000 /3600
    k_ad = st.number_input("k_ad: Kinetic of Adsorption [m^3/s/mol]", min_value=0.0000, max_value=10000.0, step=0.01)

with col3:
    M_S = st.number_input("M_S: Molecular Weight of Filtered Liquid [g/mol]", min_value=15.0, max_value=30.0, step=0.1) /1000
    rho_S = st.number_input("rho_S: Density of Filtered Liquid [g/cm^3]", min_value=0.9, max_value=1.2, step=0.05) /1000 * 1e6

V_S, n_S, N_in = preprocessing(V_Tray,eps_Tray,rho_S,M_S,V_in)

# Biomass Tracking
with col3:
    X_bio_pre = st.number_input("X_bio_pre: Biomass at the End of Last Growth Phase [kg]", value=biomasstracker, min_value=0.0, max_value=200.0, step=1.0)

with col1:
    theta_inf = st.number_input("theta_inf: Relationship between Biomass and Binding Capacity [mmol/g]", min_value=0.443, max_value=0.7, step=0.001) 

B_inf = theta_inf * X_bio_pre

adsorp_fig, ad_data = adsorp_plot(V_ad, X_REE_in, N_in, n_S, V_S, k_ad, B_inf, M_S, rho_S)

df_ad = ad_data.iloc[-1]
df_ad = df_ad.to_frame().reset_index()
df_ad.columns = ["Variable", "Value"]
df_ad = df_ad.drop(index=0)
df_ad["Unit"] = ['mol','mol','m^3','mol','mol/s','mol/mol','mol']

with col1:
    st.subheader("Variable Plots")
    st.plotly_chart(adsorp_fig)

with col3:
    st.subheader("Final Values")
    st.table(df_ad)

boundREEtracker = ad_data["n_B"][len(ad_data["n_B"])-1]

st.title("Desorption")
col1, col2 = st.columns(2)

with col1:
    V_de = st.number_input("V_de: Desired Amount of Concentrated Solution [L]", min_value=1.0, max_value=25.0, step=0.5) / 1000
    n_REE_bound = st.number_input("n_REE_bound: Bound Target Component at the Beginning of Desorption Phase	[mol]", value=boundREEtracker, min_value=0.0000, max_value=1000.0000, step=0.01)
    M_D = st.number_input("M_D: Molecular Weight of Dissolution Fluid [g/mol]", min_value=15.0, max_value=30.0, step=0.1) /1000

with col2:
    rho_D = st.number_input("rho_D: Density of Dissolution Fluid [g/cm^3]", min_value=0.9, max_value=1.2, step=0.05) /1000 * 1e6
    k_de = st.number_input("k_de: Kinetic of Desorption [1/s]", min_value=0.0000, max_value=10000.0, step=0.01)

V_D, n_D, N_in = preprocessing(V_Tray,eps_Tray,rho_D,M_D,V_in)

desorp_fig, de_data = desorp_plot(V_de,n_REE_bound,N_in,n_D,V_D,k_de,B_inf,M_D,rho_D)

df_de = de_data.iloc[-1]
df_de = df_de.to_frame().reset_index()
df_de.columns = ["Variable", "Value"]
df_de = df_de.drop(index=0)
df_de["Unit"] = ['mol','mol','m^3','mol','mol/s','mol/mol','mol']

with col1:
    st.subheader("Variable Plots")
    st.plotly_chart(desorp_fig)

with col2:
    st.subheader(" ")
    st.subheader(" ")
    st.subheader(" ")
    st.subheader(" ")
    st.subheader("Final Values")
    st.table(df_de)

# Biomass Destruction

st.title("Biomass Destruction")

t_end_ad = ad_data.iloc[-1]["Time"]
t_end_de = de_data.iloc[-1]["Time"]

d_ad = st.number_input("d_ad: Biomass Death Rate during Adsorption Phase [g/h]", min_value=50, max_value=1000, step=1) /1000 /3600
d_de = st.number_input("d_de: Biomass Death Rate during Desorption Phase [g/h]", min_value=50, max_value=10000, step=1) /1000 /3600

X_bio_new = X_bio_pre - d_ad * t_end_ad - d_de * t_end_de
st.write("Surving biomass to initialize next growth phase: " + str(round(X_bio_new,2)))